#!/bin/sh
#
# install script for ACKSYS cards xRSPCI, xRSUNI
#
# Copyright by Acksys
# JP Tosoni, 20-dec-2004
#
# depends on:
#	uname, id, lsmod, sed, cat, ls, expr, sort, awk, rm, mknod,
#	chown, test
#

LANG=C export LANG			# set standard string comparisons
DEV=/dev				# where are the device nodes
SYSTTY=/sys/class/tty			# sysfs for ttys
SYSACK=/sys/bus/pci/drivers/acksys	# sysfs for acksys driver
OWNER=root:root				# owner/group of device nodes in $DEV

# check linux version
faillinux() {
	rel=`uname -r`
	case "$rel" in
	2.6.[1-9][0-9]*) return 1;;
	*)	echo wrong linux release `uname -r`, 2.6.10 minimum required
		return 0;;
	esac
}

# check user rights
failuser() {
	case `id -u` in
	0) return 1;;
	*) echo You must be root to do this; return 0;;
	esac
}

# check whether serial is dynamic or static
failmodule() {
	mod=`lsmod|sed -n '/^8250 /s/ .*//p'`
	case "$mod" in
	8250) # linux serial is dynamic
		return 0;;
	*) # linux serial is static
		return 0;;
	esac
}

# install acksys modules
installmods() {
	ser=acksys8250.ko
	pci=acksys8250_pci.ko
	if [ ! -f $ser -o ! -f $pci ]; then
		echo You are not in the proper directory;
		return 1;
	fi
	mods=/lib/modules/`uname -r`/kernel/drivers/serial
	mkdir -p "$mods"
	mv $ser $pci $mods
	return 0
}

# install acksys script
installscript() {
	scr=acksys8250_install.sh
	if [ ! -f $ser ]; then
		echo You are not in the proper directory;
		return 1;
	fi
	rcdir=/etc/init.d
	mkdir -p "$mods"
	mv $ser $pci $mods
	return 0
}

# display Acksys ports list
listports() {
	here=`pwd`
	cd $SYSTTY
	for i in ttyS*; do
		[ -e $i/device/vendor -a -e $i/device/device ] || continue
		[ `cat $i/device/vendor` = 0x1528 ] || continue
		did=`cat $i/device/device`
		slot=`ls -l $i/device`
		slot=`expr "$slot" : '.*/\([^/]*\)..$'`
		case $did in
		0x0001 | 0x0010 | 0x0011 | 0x0012)
			echo $slot $did $i;;
		esac
	done | sort |
		sed	-e 's/ 0x0001 / UNXPCI /' \
			-e 's/ 0x0010 / 4RSPCI /' \
			-e 's/ 0x0011 / 8RSPCI /' \
			-e 's/ 0x0012 / 2RSPCI /'
	cd $here
	return 1
}

listcards() {
	listports | awk \
	'	{	if($1!=slot) {
				if(slot!="")print carte,count,slot,ttys;
				count=1;
				carte=$2;
				slot=$1;
				ttys=$3;
			} else {
				count++;
				carte=$2;
				ttys = ttys " " $3
			}
		}
		END{
			if(slot!="")print carte,count,slot,ttys;
		}'
}
Xlistcards() {
	if [ ! -e $SYSACK ]; then
		echo "Acksys driver modules not loaded - please load them"
		return 1
	fi
	here=`pwd`
	cd $SYSACK
	for i in *:*:*.*; do
		[ -e $i/vendor -a -e $i/device ] || continue
		[ `cat $i/vendor` = 0x1528 ] || continue
		did=`cat $i/device`
		slot=`ls -l $i`
		slot=`expr "$slot" : '.*/\([^/]*\)..$'`
		case $did in
			0x0001) echo $did UNXPCI $slot;;
			0x0010) echo $did 4RSPCI $slot;;
			0x0011) echo $did 8RSPCI $slot;;
			0x0012) echo $did 2RSPCI $slot;;
		esac
	done
	cd $here
}

faillistports() {
	listcards
	return 1
}

failmkdevs() {
	echo creating device nodes in /dev:
	listports |
		while read slot carte i; do
			if [ "x$slot" != "x$prev" ]; then
				echo
				echo -n $carte, slot $slot:
			fi
			prev=$slot
			min=`sed 's/.*://' $SYSTTY/$i/dev`
			maj=`sed 's/:.*//' $SYSTTY/$i/dev`
			[ -e $DEV/$i ] && rm $DEV/$i
			mknod $DEV/$i c $maj $min
			chown $OWNER $DEV/$i
			echo -n " $i"
		done
		echo
	return 1
}

set_auto_t_a() {
	if [ -x ./acksys8250turnaround ]; then
		./acksys8250turnaround "$2" $1
	else
		acksys8250turnaround "$2" $1
	fi
}

case "$1" in
'-l')	faillinux && exit 1 || echo OK;;
'-u')	failuser && exit 1 || echo OK;;
'-p')	faillistports && exit 1 || echo OK;;
'-m')	failautomod && exit 1 || echo OK;;
'-d')	failmkdevs && exit 1 || echo OK;;
'-a')	set_auto_t_a 1 "$2" && exit 1 || echo OK;;
'-n')	set_auto_t_a 0 "$2" && exit 1 || echo OK;;
'')
	if false; then
		faillinux && exit 1
		failuser && exit 1
		failmodule && exit 1
		installmods || exit 1
		faillistports && exit 1
		failautomod && exit 1
		failmkdevs && exit 1
		failrs485 && exit 1
	else
		echo type "'$0 -h'" for help
	fi
	;;
'-C')	listcards && exit 1;;
'-P')	listports && exit 1;;
'-h'|'--help'|*)
		 #non implemente $0		(check and installs all)
	cat <<-!
		usage:
		 $0 -l		(checks linux release)
		 $0 -u		(checks user rights)
		 $0 -d		(create entries like /dev/ttySxxx)
		 $0 -C		(lists recognized Acksys PCI cards)
		 $0 -P		(lists recognized Acksys PCI tty ports)
		 $0 -a /dev/ttySxxx	(set RS485 auto turnaround for port)
		 $0 -n /dev/ttySxxx	(clear RS485 auto turnaround for port)
		 $0 -h		(this help)
	!
	;;
esac
exit 0
