﻿// Flag to indicate whether the Leaflet libraries have already been loaded
export var leafletLibrariesLoaded = false;

/**
 * Reloads Leaflet and its plugins (CSS and JS) only if not already loaded.
 * If Leaflet is already present (L is defined), it will skip reloading.
 * Accepts an optional callback to run once loading is complete.
 * 
 * @param {Function} callback - Function to be called once libraries are loaded
 */
export function reloadLeafletLibraries(callback) {
    if (!leafletLibrariesLoaded) {
        // Check if the Leaflet global object is already defined
        if (typeof L === 'undefined') {
            console.log("Leaflet libraries not loaded, reloading...");

            // Remove any existing Leaflet <link> and <script> tags from the DOM
            removeExistingLeafletLinks();

            // Load Leaflet and plugin CSS files
            reloadLeafletCSS('https://unpkg.com/leaflet@1.9.4/dist/leaflet.css');
            reloadLeafletCSS('https://cdnjs.cloudflare.com/ajax/libs/leaflet.markercluster/1.5.3/MarkerCluster.css');
            reloadLeafletCSS('https://cdnjs.cloudflare.com/ajax/libs/leaflet.markercluster/1.5.3/MarkerCluster.Default.css');
            reloadLeafletCSS('https://cdn.jsdelivr.net/npm/leaflet-easybutton@2/src/easy-button.css');

            // Load JS libraries sequentially using Promises
            loadScript('https://unpkg.com/leaflet@1.9.4/dist/leaflet.js')
                .then(() => loadScript('https://cdnjs.cloudflare.com/ajax/libs/leaflet.markercluster/1.5.3/leaflet.markercluster.js'))
                .then(() => loadScript('https://cdn.jsdelivr.net/npm/leaflet-easybutton@2/src/easy-button.js'))
                .then(() => {
                    leafletLibrariesLoaded = true; // Mark libraries as loaded
                    if (callback) callback();      // Call the provided callback
                })
                .catch(error => {
                    console.error("Failed to load Leaflet libraries:", error);
                });
        } else {
            // Leaflet is already loaded, just mark as loaded and run callback
            console.log("Leaflet is already loaded.");
            leafletLibrariesLoaded = true;
            if (callback) callback();
        }
    } else {
        // Already loaded from previous call
        if (callback) callback();
    }
}

/**
 * Removes existing <link> and <script> elements for Leaflet and its plugins
 * from the document to ensure clean reloading.
 */
function removeExistingLeafletLinks() {
    var leafletLinks = [
        'https://unpkg.com/leaflet@1.9.4/dist/leaflet.css',
        'https://cdnjs.cloudflare.com/ajax/libs/leaflet.markercluster/1.5.3/MarkerCluster.css',
        'https://cdnjs.cloudflare.com/ajax/libs/leaflet.markercluster/1.5.3/MarkerCluster.Default.css',
        'https://cdn.jsdelivr.net/npm/leaflet-easybutton@2/src/easy-button.css',
        'https://unpkg.com/leaflet@1.9.4/dist/leaflet.js',
        'https://cdnjs.cloudflare.com/ajax/libs/leaflet.markercluster/1.5.3/leaflet.markercluster.js',
        'https://cdn.jsdelivr.net/npm/leaflet-easybutton@2/src/easy-button.js'
    ];

    leafletLinks.forEach(link => {
        var elements = document.querySelectorAll(`link[href="${link}"], script[src="${link}"]`);
        elements.forEach(el => el.parentNode.removeChild(el));
    });
}

/**
 * Dynamically injects a new CSS <link> into the document head.
 * 
 * @param {string} href - URL of the CSS file to load
 */
function reloadLeafletCSS(href) {
    var link = document.createElement('link');
    link.rel = 'stylesheet';
    link.href = href;
    document.head.appendChild(link);
}

/**
 * Dynamically loads a JavaScript file by injecting a <script> element.
 * Returns a Promise that resolves on success, rejects on failure.
 * 
 * @param {string} src - URL of the JavaScript file to load
 * @returns {Promise<void>}
 */
function loadScript(src) {
    return new Promise((resolve, reject) => {
        var script = document.createElement('script');
        script.src = src;
        script.onload = () => {
            console.log(src + ' loaded');
            resolve();
        };
        script.onerror = () => {
            reject(new Error(src + ' failed to load'));
        };
        document.head.appendChild(script);
    });
}