﻿import { reloadLeafletLibraries } from './leafletLoader-v1.0.1.js';

var map;        // Leaflet map instance
var marker;     // Current map marker
var lat;        // Current latitude
var lng;        // Current longitude
var buttonClicked = false;

/**
 * Stores the .NET reference so it can be invoked from JavaScript
 * @param {any} dotNetReference - Reference to a .NET Blazor object
 */
window.InitMapModalDotNetReference = function (dotNetReference) {
    window.MapModalDotNetReference = dotNetReference;
}

/**
 * Initializes the Leaflet map inside a modal for selecting a position.
 * Loads Leaflet if not already loaded.
 * 
 * @param {string} elementId - HTML element ID where the map will be rendered
 * @param {number} initialLat - Initial latitude for map view
 * @param {number} initialLng - Initial longitude for map view
 * @param {number} zoom - Zoom level
 * @param {string} productId - Identifier for the product (used in marker)
 */
window.InitializeMapModal = async function (elementId, initialLat, initialLng, zoom, productId) {
    lat = initialLat;
    lng = initialLng;

    // If a map already exists, remove it before re-initializing
    if (typeof map !== 'undefined' && map !== null) {
        console.log("Map is already loaded. Removing existing instance.");
        map.remove();
    }

    // Ensure Leaflet libraries are fully loaded
    await reloadLeafletLibrariesAsync();

    // Initialize the map centered on the given coordinates
    map = L.map(elementId).setView([lat, lng], zoom);

    // Add OpenStreetMap tile layer
    L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
        maxZoom: 19,
    }).addTo(map);

    // Handle map click to set a marker and update coordinates
    map.on('click', function (e) {
        lat = e.latlng.lat;
        lng = e.latlng.lng;
        addIconToMapModal(lat, lng, productId);
    });

    // Clear any previously set marker
    if (marker != null) {
        marker = null;
    }

    // Add a save button to the map using Leaflet.EasyButton
    var button = L.easyButton({
        states: [{
            stateName: 'save-position',
            icon: '<i class="fa fa-save"></i>', // FontAwesome icon
            title: 'Save Position',
            onClick: function (btn, map) {
                // Prevent click propagation and call .NET method with coordinates
                L.DomEvent.stopPropagation(btn);
                window.MapModalDotNetReference.invokeMethodAsync('OpenModal', lat, lng);
            }
        }]
    }).addTo(map);

    // Optional button styling
    button.button.style.backgroundColor = 'white';
    button.button.style.boxShadow = '0 0 10px rgba(0,0,0,0.9)';

    console.log("Map modal successfully initialized.");
}

/**
 * Removes the map and marker from the modal.
 * Use this when closing or resetting the modal.
 */
window.RemoveMapModal = function () {
    if (map != null) {
        map.remove();
        map = null;
    }
    if (marker != null) {
        marker = null;
    }
}

/**
 * Wrapper to turn the callback-based reloadLeafletLibraries into a Promise
 * so it can be awaited in async functions.
 * 
 * @returns {Promise<void>}
 */
function reloadLeafletLibrariesAsync() {
    return new Promise((resolve, reject) => {
        try {
            reloadLeafletLibraries(() => {
                resolve(); // Resolves when libraries are loaded
            });
        } catch (err) {
            reject(err);
        }
    });
}

/**
 * Adds or updates a marker on the map at the given coordinates.
 * Associates the marker with a productId.
 * 
 * @param {number} lat - Latitude for the marker
 * @param {number} lng - Longitude for the marker
 * @param {string} productId - Identifier associated with the marker
 */
function addIconToMapModal(lat, lng, productId) {
    if (marker) {
        // Update existing marker position
        marker.setLatLng([lat, lng]);
    } else {
        // Create a new marker and associate it with the map
        marker = L.marker([lat, lng]);
        marker.productId = productId;
        marker.addTo(map);
    }
}