﻿// --- WaveManager Service Worker ---
// Imports the list of all static assets generated by the Blazor build process.
self.importScripts('./service-worker-assets.js');

// --- Configuration ---
const cacheNamePrefix = 'wm-app-cache-v';
const cacheVersion = '4.6.0.1'; // update this version on each publish
const cacheName = `${cacheNamePrefix}${cacheVersion}`;

const offlineAssetsInclude = [
    /\.dll$/, /\.pdb$/, /\.wasm$/, /\.html$/, /\.js$/, /\.json$/,
    /\.css$/, /\.woff$/, /\.png$/, /\.jpe?g$/, /\.gif$/, /\.ico$/,
    /\.blat$/, /\.dat$/, /\.svg$/, /\.webmanifest$/
];
const offlineAssetsExclude = [
    /^index\.html$/, /^sw-registrator\.js$/, /^service-worker\.js$/
];

// --- INSTALL ---
// Runs when a new service worker is first downloaded.
// Caches all required static assets so the app can work offline.
self.addEventListener('install', event => {
    console.info(`[SW] Install → ${cacheName}`);
    event.waitUntil(onInstall());
});

async function onInstall() {
    // Force installation immediately
    self.skipWaiting();

    const assetsToCache = self.assetsManifest.assets
        .filter(a => offlineAssetsInclude.some(rx => rx.test(a.url)))
        .filter(a => !offlineAssetsExclude.some(rx => rx.test(a.url)))
        .map(a => new Request(a.url, { integrity: a.hash, cache: 'no-cache' }));

    const cache = await caches.open(cacheName);
    await cache.addAll(assetsToCache);

    console.info(`[SW] Cached ${assetsToCache.length} assets.`);
}

// --- ACTIVATE ---
// Runs after installation. Cleans old caches and takes control
// of all open tabs to ensure the newest version is used.
self.addEventListener('activate', event => {
    console.info('[SW] Activate → cleaning old caches');
    event.waitUntil(onActivate());
});

async function onActivate() {
    const keys = await caches.keys();

    await Promise.all(
        keys
            .filter(k => k.startsWith(cacheNamePrefix) && k !== cacheName)
            .map(k => {
                console.info(`[SW] Deleting old cache: ${k}`);
                return caches.delete(k);
            })
    );

    // Take control immediately of all open tabs
    await self.clients.claim();
    console.info('[SW] Now controlling all clients.');
}

// --- FETCH ---
// Intercepts all network requests. Serves cached files when possible,
// bypasses cache for API calls, and ensures offline support.
self.addEventListener('fetch', event => {
    event.respondWith(onFetch(event.request));
});

async function onFetch(req) {
    const url = req.url;

    // Never cache API calls
    if (url.includes('/api/')) {
        return fetch(req);
    }

    const cache = await caches.open(cacheName);

    // Navigation handling
    if (req.mode === 'navigate') {
        const cachedIndex = await cache.match('index.html');
        return cachedIndex || fetch(req);
    }

    // Cache-first strategy
    const cachedResponse = await cache.match(req);
    if (cachedResponse) return cachedResponse;

    const networkResponse = await fetch(req);

    if (networkResponse && networkResponse.status === 200) {
        const isAsset = offlineAssetsInclude.some(rx => rx.test(url))
            && !offlineAssetsExclude.some(rx => rx.test(url));

        if (isAsset) {
            cache.put(req.url, networkResponse.clone());
        }
    }

    return networkResponse;
}/* Manifest version: yXW0K2CA */
