#!/bin/bash

# This script configures the Mosquitto service for secure, non-root operation
# while allowing it to bind to privileged ports (e.g., 443) using file capabilities.

# Rationale for configuration steps:
# 1. setcap: Grants the /usr/sbin/mosquitto executable the CAP_NET_BIND_SERVICE
#           capability, allowing it to bind to ports < 1024 (like 443) without
#           requiring the process to run as root permanently.
# 2. User/Group: Ensures the service process drops root privileges immediately
#                after binding the port, running securely as the 'mosquitto' user.
# 3. Custom Config: Copies the WaveManager-specific Mosquitto configuration file.

# Define Variables
ACKSYS_WAVEMANAGER_MOSQUITTO_CONFIGURATION_FILE="/opt/acksys/etc/wm-mosquitto.conf"
MOSQUITTO_SERVICE_NAME="mosquitto.service"
MOSQUITTO_USER="mosquitto"
MOSQUITTO_CONFIGURATION_DIR="/etc/mosquitto/conf.d/"
MOSQUITTO_SERVICE_FILE=$(systemctl show "$MOSQUITTO_SERVICE_NAME" --property=FragmentPath --value)

# Step 01 : Check variables
# Check mosquitto service file path
if [ -z "$MOSQUITTO_SERVICE_FILE" ]; then
    echo "Error: Service file for $MOSQUITTO_SERVICE_NAME not found by systemctl. Cannot configure."
    exit 1
fi
echo "Configuring Mosquitto service file: $MOSQUITTO_SERVICE_FILE"

# Check mosquitto user
if id "$MOSQUITTO_USER" >/dev/null 2>&1; then
    echo "Dedicated user '$MOSQUITTO_USER' exists. Proceeding with configuration."
    USER_EXISTS=true
else
    echo "Warning: Dedicated user '$MOSQUITTO_USER' not found. Skipping User/Group configuration."
    USER_EXISTS=false
fi

# Step 02 : Stop the mosquitto service
systemctl stop mosquitto

# Step 03 : Add the wm mosquitto conf file
cp "$ACKSYS_WAVEMANAGER_MOSQUITTO_CONFIGURATION_FILE" "$MOSQUITTO_CONFIGURATION_DIR"

# Step 04 : Set capabilities to use port < 1024
setcap CAP_NET_BIND_SERVICE=+ep /usr/sbin/mosquitto

# Step 05 : Configure the mosquitto service to allow using port < 1024
if [ "$USER_EXISTS" = true ]; then
    # Ensure 'User=mosquitto' is set
    if ! grep -q '^User=' "$MOSQUITTO_SERVICE_FILE"; then
        sed -i "/^\[Service\]/a User=$MOSQUITTO_USER" "$MOSQUITTO_SERVICE_FILE"
        echo "Inserted User=$MOSQUITTO_USER"
    else
        sed -i "s/^\(User=\).*$/User=$MOSQUITTO_USER/" "$MOSQUITTO_SERVICE_FILE"
        echo "Updated User to $MOSQUITTO_USER"
    fi

    # Ensure 'Group=mosquitto' is set 
    if ! grep -q '^Group=' "$MOSQUITTO_SERVICE_FILE"; then
        sed -i "/^\[Service\]/a Group=$MOSQUITTO_USER" "$MOSQUITTO_SERVICE_FILE"
        echo "Inserted Group=$MOSQUITTO_USER"
    else
        sed -i "s/^\(Group=\).*$/Group=$MOSQUITTO_USER/" "$MOSQUITTO_SERVICE_FILE"
        echo "Updated Group to $MOSQUITTO_USER"
    fi
else
    echo "Skipping setting User and Group. Service will run as root temporarily."
fi

# Step 06 : Reload and Restart
echo "Reloading systemd daemon..."
systemctl daemon-reload
echo "Restarting mosquitto service..."
# Restart is sufficient; it stops and starts.
systemctl restart mosquitto

exit 0
