﻿import { reloadLeafletLibraries } from './leafletLoader-v1.0.1.js';

(() => {
    // Global variables
    let map = null;
    let markers = null;
    let openPopupId = null;
    let popupWasClosed = false;
    let manualClose = false;

    // Cluster and zoom state tracking
    const clusterZoomLevel = 16;
    let clusterClickedZoom = null;
    let clusterClicked = false;
    const initialZoomLevel = 10;
    let lastZoomedProduct = null;
    let lastProductList = [];
    let clusterZoomTriggered = false;

    // Initializes the .NET reference for Blazor interop
    window.InitMapDotNetReference = (dotNetReference) => {
        window.MapDotNetReference = dotNetReference;
    };

    // Asynchronously reload Leaflet libraries
    const reloadLeafletLibrariesAsync = () =>
        new Promise((resolve, reject) => {
            try {
                reloadLeafletLibraries(resolve);
            } catch (err) {
                reject(err);
            }
        });

    // Main function to create and initialize the map
    window.CreateMap = async () => {
        try {
            // Remove existing map instance if already loaded
            if (map) {
                console.log("Map already loaded.");
                map.remove();
            }

            // Load Leaflet libraries
            await reloadLeafletLibrariesAsync();

            // Initialize the map
            map = L.map('WmMap', {
                zoomControl: false,
                minZoom: 2,
                maxBoundsViscosity: 1.0,
                worldCopyJump: true
            });

            // Add base tile layer (OpenStreetMap)
            L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
                maxZoom: 19,
                maxNativeZoom: 19,
                crossOrigin: true
            }).addTo(map);

            // Create marker cluster group
            markers = L.markerClusterGroup({
                spiderfyOnMaxZoom: false,
                maxClusterRadius: 19,
                disableClusteringAtZoom: 19,
                iconCreateFunction: (cluster) => {
                    const childMarkers = cluster.getAllChildMarkers();
                    const isSelected = childMarkers.some(marker => marker.options.isSelected);
                    const className = `marker-cluster${isSelected ? ' selected' : ''}`;
                    const countStyle = isSelected ? 'animation: blinker 1s linear infinite;' : '';

                    return new L.DivIcon({
                        html: `<style>@keyframes blinker { 50% { opacity: 0; } }</style><div><span style="${countStyle}">${cluster.getChildCount()}</span></div>`,
                        className,
                        iconSize: new L.Point(40, 40)
                    });
                }
            });

            // Add UI controls
            L.control.zoom({ position: 'topright' }).addTo(map);
            L.control.scale({ position: 'bottomright' }).addTo(map);

            // Add custom buttons
            createButtons(map);

            // Register event handlers
            map.on('popupclose', handlePopupClose);
            map.addLayer(markers);

            // Center map either on content bounds or default position
            const bounds = new L.LatLngBounds();
            if (bounds.isValid()) {
                map.fitBounds(bounds.pad(0.1), { maxZoom: 14 });
            } else {
                map.setView([0, 0], 2); // Default world view
            }
        } catch (e) {
            console.error("Error creating map:", e);
        }
    };

    // Creates custom Leaflet easy buttons (Save View / Show-Hide Table)
    const createButtons = (map) => {
        // Button to save the current view
        L.easyButton({
            id: 'button-save',
            position: 'topright',
            type: 'replace',
            leafletClasses: true,
            states: [{
                stateName: 'get-center',
                onClick: () => window.MapDotNetReference.invokeMethodAsync('OnAddMapViewRequested'),
                title: 'Save the view',
                icon: 'fa-save'
            }]
        }).addTo(map);

        // Toggle button to show/hide saved views table
        L.easyButton({
            id: 'button-table',
            position: 'topright',
            type: 'replace',
            leafletClasses: true,
            states: [{
                stateName: 'show-table',
                onClick: (button) => {
                    window.MapDotNetReference.invokeMethodAsync('GetAllUserMapView')
                        .then(data => {
                            const table = document.getElementById("savedViews");
                            const tbody = table.querySelector('tbody');
                            tbody.innerHTML = '';
                            data.forEach(view => createTableRow(view, map, tbody));
                            table.classList.add("show");
                            button.state('hide-table');
                            window.isTableVisible = true;
                        });
                },
                title: 'Show Views',
                icon: 'fa-chevron-up'
            }, {
                stateName: 'hide-table',
                onClick: (button) => {
                    document.getElementById("savedViews").classList.remove("show");
                    button.state('show-table');
                    window.isTableVisible = false;
                },
                title: 'Hide Views',
                icon: 'fa-chevron-down'
            }]
        }).addTo(map);
    };

    // Handles popup close event
    const handlePopupClose = () => {
        if (map._popup?._source) {
            openPopupId = null;
            popupWasClosed = true;
            manualClose = true;
        } else {
            manualClose = false;
        }
    };

    // Returns the current zoom level
    window.GetZoom = () => map?.getZoom() ?? null;

    // Returns the current center coordinates
    window.GetCenter = () => map?.getCenter() ?? null;

    // Refreshes the markers on the map based on incoming product data
    window.RefreshProductMarkers = (productsWithContent) => {
        if (!map || !markers || !productsWithContent) {
            console.error("Map or markers is undefined.");
            return;
        }

        let wasOpen = map._popup?._source && !popupWasClosed;
        if (wasOpen) openPopupId = map._popup._source._id;

        const currentZoom = map.getZoom();
        const currentCenter = map.getCenter();
        const markerList = {};

        // Index existing markers by ID
        markers.eachLayer(marker => markerList[marker._id] = marker);
        const previousProductCount = Object.keys(markerList).length;
        const bounds = new L.LatLngBounds();
        let newProductZoomed = false;

        // Loop over all products and update or create markers
        productsWithContent.forEach(({ product, htmlContent, iconContent }) => {
            const customIcon = L.divIcon({ className: 'custom-icon', html: iconContent });
            const existingMarker = markerList[product.uniqueId];
            let marker;

            if (existingMarker) {
                existingMarker.options.isSelected = product.isSelected;
                const { lat, lng } = existingMarker._latlng;
                const { latitude, longitude } = product.gnssStatus;

                if (lat === parseFloat(latitude) && lng === parseFloat(longitude)) {
                    existingMarker.setIcon(customIcon);
                    existingMarker.setPopupContent(htmlContent);
                    if (wasOpen && openPopupId === product.uniqueId && !manualClose) {
                        existingMarker.openPopup();
                    }
                    delete markerList[product.uniqueId];
                    marker = existingMarker;
                } else {
                    markers.removeLayer(existingMarker);
                    marker = createMarker(product, customIcon, htmlContent);
                }
            } else {
                marker = createMarker(product, customIcon, htmlContent);
                newProductZoomed = true;
            }

            bounds.extend(marker.getLatLng());
        });

        // Remove outdated markers
        Object.values(markerList).forEach(marker => markers.removeLayer(marker));

        // Adjust map view if necessary
        if (bounds.isValid() && productsWithContent.length > 0) {
            const currentProductCount = productsWithContent.length;
            if (previousProductCount !== currentProductCount || newProductZoomed) {
                map.fitBounds(bounds.pad(0.1), { maxZoom: 14 });
            } else {
                map.setView(currentCenter, currentZoom);
            }
        } else {
            map.setView([0, 0], 2);
        }

        popupWasClosed = false;
    };

    // Creates a Leaflet marker with popup
    const createMarker = (product, icon, popupContent) => {
        const marker = L.marker([product.gnssStatus.latitude, product.gnssStatus.longitude], {
            icon,
            isSelected: product.isSelected
        });
        marker._id = product.uniqueId;
        marker.bindPopup(popupContent);
        markers.addLayer(marker);
        if (openPopupId === product.uniqueId && !manualClose) {
            marker.openPopup();
        }
        return marker;
    };

    // Refreshes the saved map views table
    window.RefreshMapViewTable = () => {
        window.MapDotNetReference.invokeMethodAsync('GetAllUserMapView')
            .then(data => {
                const tbody = document.querySelector("#savedViews tbody");
                tbody.innerHTML = '';
                data.forEach(view => createTableRow(view, map, tbody));
            });
    };

    // Appends a row to the saved views table
    const createTableRow = (view, map, tbody) => {
        const row = tbody.insertRow();
        const [cell1, cell2, cell3, actionCell] = [0, 1, 2, 3].map(() => row.insertCell());

        cell1.innerText = view.name;
        cell2.innerText = view.longitude;
        cell3.innerText = view.latitude;
        actionCell.innerHTML = '<i class="fas fa-times-circle delete-button" style="color:red"></i>';

        // Set view on row click
        row.onclick = () => map.setView([view.latitude, view.longitude], view.zoom);

        // Handle delete click
        actionCell.querySelector('.delete-button').onclick = (e) => {
            e.stopPropagation();
            window.MapDotNetReference.invokeMethodAsync('OnDeleteMapViewRequested', view.id);
        };
    };

    // Request .NET to show product detail panel
    window.OnProductDetailRequested = (productId) => {
        window.MapDotNetReference.invokeMethodAsync('OnProductDetailRequested', productId);
    };

    // Forces map to re-calculate its size
    window.RefreshMapSize = () => {
        for (let i = 0; i < 3; i++) {
            map.invalidateSize();
        }
    };
})();